<?php
/**
 * MAXEDR10 - MAX ERP 10
 * Model: Cliente
 */

namespace Maxedr10\Models;

use Maxedr10\Database\Database;

class Cliente {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    /**
     * Lista todos os clientes de uma empresa
     */
    public function listar($empresaId, $filtros = []) {
        $sql = "SELECT * FROM clientes WHERE empresa_id = ? AND ativo = 1";
        $params = [$empresaId];
        
        // Filtro por nome
        if (!empty($filtros['nome'])) {
            $sql .= " AND nome LIKE ?";
            $params[] = '%' . $filtros['nome'] . '%';
        }
        
        $sql .= " ORDER BY nome";
        
        return $this->db->query($sql, $params);
    }
    
    /**
     * Busca um cliente por ID
     */
    public function buscarPorId($id, $empresaId) {
        $sql = "SELECT * FROM clientes WHERE id = ? AND empresa_id = ?";
        return $this->db->queryOne($sql, [$id, $empresaId]);
    }
    
    /**
     * Cria um novo cliente
     */
    public function criar($dados) {
        $sql = "INSERT INTO clientes (
                    empresa_id, nome, cpf_cnpj, tipo, telefone, email, 
                    endereco, numero, complemento, bairro, cidade, estado, cep, observacoes
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        
        $params = [
            $dados['empresa_id'],
            $dados['nome'],
            $dados['cpf_cnpj'] ?? null,
            $dados['tipo'] ?? 'fisica',
            $dados['telefone'] ?? null,
            $dados['email'] ?? null,
            $dados['endereco'] ?? null,
            $dados['numero'] ?? null,
            $dados['complemento'] ?? null,
            $dados['bairro'] ?? null,
            $dados['cidade'] ?? null,
            $dados['estado'] ?? null,
            $dados['cep'] ?? null,
            $dados['observacoes'] ?? null
        ];
        
        if ($this->db->execute($sql, $params)) {
            return $this->db->lastInsertId();
        }
        
        return false;
    }
    
    /**
     * Atualiza um cliente
     */
    public function atualizar($id, $empresaId, $dados) {
        $sql = "UPDATE clientes SET 
                    nome = ?, cpf_cnpj = ?, tipo = ?, telefone = ?, email = ?,
                    endereco = ?, numero = ?, complemento = ?, bairro = ?, 
                    cidade = ?, estado = ?, cep = ?, observacoes = ?
                WHERE id = ? AND empresa_id = ?";
        
        $params = [
            $dados['nome'],
            $dados['cpf_cnpj'] ?? null,
            $dados['tipo'] ?? 'fisica',
            $dados['telefone'] ?? null,
            $dados['email'] ?? null,
            $dados['endereco'] ?? null,
            $dados['numero'] ?? null,
            $dados['complemento'] ?? null,
            $dados['bairro'] ?? null,
            $dados['cidade'] ?? null,
            $dados['estado'] ?? null,
            $dados['cep'] ?? null,
            $dados['observacoes'] ?? null,
            $id,
            $empresaId
        ];
        
        return $this->db->execute($sql, $params);
    }
    
    /**
     * Deleta um cliente (soft delete)
     */
    public function deletar($id, $empresaId) {
        $sql = "UPDATE clientes SET ativo = 0 WHERE id = ? AND empresa_id = ?";
        return $this->db->execute($sql, [$id, $empresaId]);
    }
}
