<?php
/**
 * MAXEDR10 - MAX ERP 10
 * Lançamentos Financeiros
 */

require_once __DIR__ . '/../config/config.php';

$pageTitle = 'Lançamentos Financeiros';
include __DIR__ . '/includes/header.php';
include __DIR__ . '/includes/sidebar.php';
?>

<div class="container-fluid">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h2><i class="fas fa-money-bill-wave"></i> Lançamentos Financeiros</h2>
        <button class="btn btn-primary" onclick="novoLancamento()">
            <i class="fas fa-plus"></i> Novo Lançamento
        </button>
    </div>
    
    <!-- Filtros -->
    <div class="card mb-4">
        <div class="card-body">
            <div class="row">
                <div class="col-md-3">
                    <label class="form-label">Data Inicial</label>
                    <input type="date" class="form-control" id="filtroDataInicial">
                </div>
                <div class="col-md-3">
                    <label class="form-label">Data Final</label>
                    <input type="date" class="form-control" id="filtroDataFinal">
                </div>
                <div class="col-md-3">
                    <label class="form-label">Tipo</label>
                    <select class="form-control" id="filtroTipo">
                        <option value="">Todos</option>
                        <option value="entrada">Entrada</option>
                        <option value="saida">Saída</option>
                    </select>
                </div>
                <div class="col-md-3">
                    <label class="form-label">Categoria</label>
                    <select class="form-control" id="filtroCategoria">
                        <option value="">Todas</option>
                    </select>
                </div>
            </div>
            <div class="row mt-3">
                <div class="col-md-12">
                    <button class="btn btn-primary" onclick="carregarLancamentos()">
                        <i class="fas fa-search"></i> Filtrar
                    </button>
                    <button class="btn btn-secondary" onclick="limparFiltros()">
                        <i class="fas fa-times"></i> Limpar
                    </button>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Tabela de Lançamentos -->
    <div class="table-responsive">
        <table class="table">
            <thead>
                <tr>
                    <th>Data</th>
                    <th>Tipo</th>
                    <th>Descrição</th>
                    <th>Categoria</th>
                    <th>Valor</th>
                    <th>Forma Pgto</th>
                    <th>Ações</th>
                </tr>
            </thead>
            <tbody id="tabelaLancamentos">
                <tr>
                    <td colspan="7" class="text-center">Carregando...</td>
                </tr>
            </tbody>
        </table>
    </div>
</div>

<!-- Modal de Lançamento -->
<div class="modal" id="modalLancamento">
    <div class="modal-content">
        <div class="modal-header">
            <h5 id="modalTitulo">Novo Lançamento</h5>
            <button class="close" onclick="fecharModal()">&times;</button>
        </div>
        <div class="modal-body">
            <form id="formLancamento">
                <input type="hidden" id="lancamentoId">
                
                <div class="row">
                    <div class="col-md-6">
                        <div class="form-group">
                            <label class="form-label">Data *</label>
                            <input type="date" class="form-control" id="data" required>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="form-group">
                            <label class="form-label">Tipo *</label>
                            <select class="form-control" id="tipo" required onchange="carregarCategoriasPorTipo()">
                                <option value="">Selecione</option>
                                <option value="entrada">Entrada</option>
                                <option value="saida">Saída</option>
                            </select>
                        </div>
                    </div>
                </div>
                
                <div class="form-group">
                    <label class="form-label">Descrição *</label>
                    <input type="text" class="form-control" id="descricao" required>
                </div>
                
                <div class="row">
                    <div class="col-md-6">
                        <div class="form-group">
                            <label class="form-label">Valor *</label>
                            <input type="number" step="0.01" class="form-control" id="valor" required>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="form-group">
                            <label class="form-label">Categoria</label>
                            <select class="form-control" id="categoria_id">
                                <option value="">Selecione</option>
                            </select>
                        </div>
                    </div>
                </div>
                
                <div class="form-group">
                    <label class="form-label">Forma de Pagamento</label>
                    <select class="form-control" id="forma_pagamento">
                        <option value="dinheiro">Dinheiro</option>
                        <option value="pix">PIX</option>
                        <option value="cartao_credito">Cartão de Crédito</option>
                        <option value="cartao_debito">Cartão de Débito</option>
                        <option value="boleto">Boleto</option>
                        <option value="transferencia">Transferência</option>
                        <option value="outros">Outros</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label class="form-label">Observações</label>
                    <textarea class="form-control" id="observacoes" rows="3"></textarea>
                </div>
            </form>
        </div>
        <div class="modal-footer">
            <button class="btn btn-secondary" onclick="fecharModal()">Cancelar</button>
            <button class="btn btn-primary" onclick="salvarLancamento()">Salvar</button>
        </div>
    </div>
</div>

<script>
let categorias = [];

// Carregar categorias
async function carregarCategorias() {
    try {
        const response = await fetch(`${API_URL}/categorias/index.php`);
        const data = await response.json();
        
        if (data.success) {
            categorias = data.data;
            
            // Preencher select de filtro
            const selectFiltro = document.getElementById('filtroCategoria');
            selectFiltro.innerHTML = '<option value="">Todas</option>';
            data.data.forEach(cat => {
                selectFiltro.innerHTML += `<option value="${cat.id}">${cat.nome}</option>`;
            });
        }
    } catch (error) {
        console.error('Erro ao carregar categorias:', error);
    }
}

// Carregar categorias por tipo no modal
function carregarCategoriasPorTipo() {
    const tipo = document.getElementById('tipo').value;
    const select = document.getElementById('categoria_id');
    
    select.innerHTML = '<option value="">Selecione</option>';
    
    if (tipo) {
        const categoriasFiltr adas = categorias.filter(c => c.tipo === tipo || c.tipo === 'ambos');
        categoriasFiltr adas.forEach(cat => {
            select.innerHTML += `<option value="${cat.id}">${cat.nome}</option>`;
        });
    }
}

// Carregar lançamentos
async function carregarLancamentos() {
    try {
        const filtros = new URLSearchParams({
            data_inicial: document.getElementById('filtroDataInicial').value,
            data_final: document.getElementById('filtroDataFinal').value,
            tipo: document.getElementById('filtroTipo').value,
            categoria_id: document.getElementById('filtroCategoria').value
        });
        
        const response = await fetch(`${API_URL}/lancamentos/index.php?${filtros}`);
        const data = await response.json();
        
        const tbody = document.getElementById('tabelaLancamentos');
        
        if (data.success && data.data.length > 0) {
            tbody.innerHTML = data.data.map(lanc => `
                <tr>
                    <td>${formatDate(lanc.data)}</td>
                    <td>
                        <span class="badge badge-${lanc.tipo === 'entrada' ? 'success' : 'danger'}">
                            ${lanc.tipo === 'entrada' ? 'Entrada' : 'Saída'}
                        </span>
                    </td>
                    <td>${lanc.descricao}</td>
                    <td>
                        ${lanc.categoria_nome ? `<span style="color: ${lanc.categoria_cor}">${lanc.categoria_nome}</span>` : '-'}
                    </td>
                    <td>${formatCurrency(lanc.valor)}</td>
                    <td>${lanc.forma_pagamento}</td>
                    <td>
                        <button class="btn btn-sm btn-primary" onclick="editarLancamento(${lanc.id})">
                            <i class="fas fa-edit"></i>
                        </button>
                        <button class="btn btn-sm btn-danger" onclick="deletarLancamento(${lanc.id})">
                            <i class="fas fa-trash"></i>
                        </button>
                    </td>
                </tr>
            `).join('');
        } else {
            tbody.innerHTML = '<tr><td colspan="7" class="text-center">Nenhum lançamento encontrado</td></tr>';
        }
    } catch (error) {
        console.error('Erro ao carregar lançamentos:', error);
        showAlert('Erro ao carregar lançamentos', 'danger');
    }
}

// Novo lançamento
function novoLancamento() {
    document.getElementById('formLancamento').reset();
    document.getElementById('lancamentoId').value = '';
    document.getElementById('modalTitulo').textContent = 'Novo Lançamento';
    document.getElementById('data').value = new Date().toISOString().split('T')[0];
    document.getElementById('modalLancamento').classList.add('show');
}

// Editar lançamento
async function editarLancamento(id) {
    try {
        const response = await fetch(`${API_URL}/lancamentos/index.php`);
        const data = await response.json();
        
        if (data.success) {
            const lanc = data.data.find(l => l.id == id);
            
            if (lanc) {
                document.getElementById('lancamentoId').value = lanc.id;
                document.getElementById('data').value = lanc.data;
                document.getElementById('tipo').value = lanc.tipo;
                document.getElementById('descricao').value = lanc.descricao;
                document.getElementById('valor').value = lanc.valor;
                document.getElementById('categoria_id').value = lanc.categoria_id || '';
                document.getElementById('forma_pagamento').value = lanc.forma_pagamento;
                document.getElementById('observacoes').value = lanc.observacoes || '';
                
                carregarCategoriasPorTipo();
                
                document.getElementById('modalTitulo').textContent = 'Editar Lançamento';
                document.getElementById('modalLancamento').classList.add('show');
            }
        }
    } catch (error) {
        console.error('Erro ao carregar lançamento:', error);
    }
}

// Salvar lançamento
async function salvarLancamento() {
    const id = document.getElementById('lancamentoId').value;
    const dados = {
        data: document.getElementById('data').value,
        tipo: document.getElementById('tipo').value,
        descricao: document.getElementById('descricao').value,
        valor: parseFloat(document.getElementById('valor').value),
        categoria_id: document.getElementById('categoria_id').value || null,
        forma_pagamento: document.getElementById('forma_pagamento').value,
        observacoes: document.getElementById('observacoes').value
    };
    
    try {
        const url = id ? `${API_URL}/lancamentos/index.php?id=${id}` : `${API_URL}/lancamentos/index.php`;
        const method = id ? 'PUT' : 'POST';
        
        const response = await fetch(url, {
            method: method,
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(dados)
        });
        
        const data = await response.json();
        
        if (data.success) {
            showAlert(data.message, 'success');
            fecharModal();
            carregarLancamentos();
        } else {
            showAlert(data.message, 'danger');
        }
    } catch (error) {
        console.error('Erro ao salvar:', error);
        showAlert('Erro ao salvar lançamento', 'danger');
    }
}

// Deletar lançamento
async function deletarLancamento(id) {
    if (!confirmAction('Deseja realmente deletar este lançamento?')) return;
    
    try {
        const response = await fetch(`${API_URL}/lancamentos/index.php?id=${id}`, {
            method: 'DELETE'
        });
        
        const data = await response.json();
        
        if (data.success) {
            showAlert(data.message, 'success');
            carregarLancamentos();
        } else {
            showAlert(data.message, 'danger');
        }
    } catch (error) {
        console.error('Erro ao deletar:', error);
        showAlert('Erro ao deletar lançamento', 'danger');
    }
}

// Fechar modal
function fecharModal() {
    document.getElementById('modalLancamento').classList.remove('show');
}

// Limpar filtros
function limparFiltros() {
    document.getElementById('filtroDataInicial').value = '';
    document.getElementById('filtroDataFinal').value = '';
    document.getElementById('filtroTipo').value = '';
    document.getElementById('filtroCategoria').value = '';
    carregarLancamentos();
}

// Inicializar
document.addEventListener('DOMContentLoaded', () => {
    carregarCategorias();
    carregarLancamentos();
});
</script>

<?php include __DIR__ . '/includes/footer.php'; ?>
