<?php
/**
 * MAXEDR10 - MAX ERP 10
 * API de Lançamentos Financeiros
 * 
 * GET    /api/lancamentos/index.php - Lista lançamentos
 * POST   /api/lancamentos/index.php - Cria lançamento
 * PUT    /api/lancamentos/index.php?id=X - Atualiza lançamento
 * DELETE /api/lancamentos/index.php?id=X - Deleta lançamento
 */

require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../src/autoload.php';

use Maxedr10\Models\Lancamento;

setJsonHeaders();

// Verificar autenticação
if (!isAuthenticated()) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Não autenticado']);
    exit;
}

$method = $_SERVER['REQUEST_METHOD'];
$user = getAuthUser();
$empresaId = $user['empresa_id'];

try {
    $lancamentoModel = new Lancamento();
    
    // ========== GET - Listar lançamentos ==========
    if ($method === 'GET') {
        $filtros = [
            'data_inicial' => $_GET['data_inicial'] ?? null,
            'data_final' => $_GET['data_final'] ?? null,
            'tipo' => $_GET['tipo'] ?? null,
            'categoria_id' => $_GET['categoria_id'] ?? null
        ];
        
        $lancamentos = $lancamentoModel->listar($empresaId, $filtros);
        
        echo json_encode([
            'success' => true,
            'data' => $lancamentos
        ]);
    }
    
    // ========== POST - Criar lançamento ==========
    elseif ($method === 'POST') {
        $data = json_decode(file_get_contents('php://input'), true);
        
        // Validações básicas
        if (empty($data['data']) || empty($data['tipo']) || empty($data['descricao']) || empty($data['valor'])) {
            http_response_code(400);
            echo json_encode([
                'success' => false,
                'message' => 'Campos obrigatórios: data, tipo, descricao, valor'
            ]);
            exit;
        }
        
        $data['empresa_id'] = $empresaId;
        $data['usuario_id'] = $user['id'];
        
        $id = $lancamentoModel->criar($data);
        
        if ($id) {
            echo json_encode([
                'success' => true,
                'message' => 'Lançamento criado com sucesso',
                'id' => $id
            ]);
        } else {
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'message' => 'Erro ao criar lançamento'
            ]);
        }
    }
    
    // ========== PUT - Atualizar lançamento ==========
    elseif ($method === 'PUT') {
        $id = $_GET['id'] ?? null;
        
        if (!$id) {
            http_response_code(400);
            echo json_encode([
                'success' => false,
                'message' => 'ID do lançamento não informado'
            ]);
            exit;
        }
        
        $data = json_decode(file_get_contents('php://input'), true);
        
        if ($lancamentoModel->atualizar($id, $empresaId, $data)) {
            echo json_encode([
                'success' => true,
                'message' => 'Lançamento atualizado com sucesso'
            ]);
        } else {
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'message' => 'Erro ao atualizar lançamento'
            ]);
        }
    }
    
    // ========== DELETE - Deletar lançamento ==========
    elseif ($method === 'DELETE') {
        $id = $_GET['id'] ?? null;
        
        if (!$id) {
            http_response_code(400);
            echo json_encode([
                'success' => false,
                'message' => 'ID do lançamento não informado'
            ]);
            exit;
        }
        
        if ($lancamentoModel->deletar($id, $empresaId)) {
            echo json_encode([
                'success' => true,
                'message' => 'Lançamento deletado com sucesso'
            ]);
        } else {
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'message' => 'Erro ao deletar lançamento'
            ]);
        }
    }
    
    else {
        http_response_code(405);
        echo json_encode(['success' => false, 'message' => 'Método não permitido']);
    }
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Erro no servidor: ' . $e->getMessage()
    ]);
}
